//
//  BaseViewController.swift
//  Demo iOS
//
//  Created by Benoit Vasseur on 28/06/2022.
//

import UIKit

class BaseViewController: UIViewController, UITextFieldDelegate {

    override func viewDidLoad() {
        super.viewDidLoad()

        if JailbreakService.isDeviceJailbroken() {
            self.showAlert(title: "Jailbreak", message: "This app can't be used on a jailbroken device", button1: (name: "Ok", action: {
                _ in exit(1)
            }), button2: nil)
        }
        // Do any additional setup after loading the view.
    }

    func textFieldShouldReturn(_ textField: UITextField) -> Bool {
        textField.resignFirstResponder()
        return false
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        NotificationCenter.default.removeObserver(self)
    }
    
    func showAlert(title: String, message: String) {
        let alert = UIAlertController(title: title, message: message, preferredStyle: .alert)
        alert.addAction(UIAlertAction(title: "OK", style: .default, handler: nil))
        
        self.present(alert, animated: true, completion: nil)
    }
    
    func showErrorFor(code: Int32) {
        showAlert(title: "Error", message: "\(InweboService.sharedInstance.errorMessageFor(code: code)), code: \(code)")
    }
    
    typealias ButtonAction = (name: String, action: ((UIAlertAction) -> Void)?)
    func showAlert(title: String, message: String, button1: ButtonAction, button2: ButtonAction?) {
        let alert = UIAlertController(title: title, message: message, preferredStyle: .alert)
        alert.addAction(UIAlertAction(title: button1.name, style: .default, handler: button1.action))
        if let button2 = button2 {
            alert.addAction(UIAlertAction(title: button2.name, style: .cancel, handler: button2.action))
        }
        self.present(alert, animated: true, completion: nil)
    }
    
    func showAlertWihtPinfield(title: String, message: String, button1: (name: String, action: ((String?) -> Void)?), button2: ButtonAction?) {
        let alert = UIAlertController(title: title, message: message, preferredStyle: .alert)
        alert.addAction(UIAlertAction(title: button1.name, style: .default, handler: { action in
            guard let textField =  alert.textFields?.first else {
                button1.action?(nil)
                return
            }
            button1.action?(textField.text)
        }))
        if let button2 = button2 {
            alert.addAction(UIAlertAction(title: button2.name, style: .cancel, handler: button2.action))
        }
        
        alert.addTextField { textfield in
            textfield.placeholder = "PIN"
            textfield.isSecureTextEntry = true
        }
        
        self.present(alert, animated: true, completion: nil)
    }

    // MARK: - Notifications
    
    func showPushOTPAuthentication(service: String?, alias: String) {
        let message = service == nil ? "You received an authentication request. Proceed ?" : "You received an authentication request from \(service!). Proceed ?"
        showAlert(title: "Notification", message: message, button1: (name: "Ok", {
            _ in self.sendPushOTP(service: service, alias: alias)
        }), button2: (name: "Cancel", {
            _ in self.sendOTP(alias: alias, pin: "", ok: false, isBio: false)
        }))
    }
    
    func askForPinForPushOTP(alias: String) {
        self.showAlertWihtPinfield(title: "Pin", message: "Enter your pin code", button1: (name: "OK", action:
                                                                                            {
            pin in
                if let pin = pin {
                    self.sendOTP(alias: alias, pin: pin, ok: true, isBio: false)
                } else {
                    self.showAlert(title: "Pin", message: "Incorrect PIN")
                }
        }), button2: (name: "Cancel", action: nil))
    }
    
    func sendPushOTP(service: String?, alias: String) {
        DispatchQueue.global().async {
            let result = InweboService.sharedInstance.startPushOTP(alias: alias)
            DispatchQueue.main.async {
                if result == IW_ERR_OK {
                    let pinMode = InweboService.sharedInstance.getPinMode()
                    if pinMode == IW_PINMODE_NONE {
                        self.sendOTP(alias: alias, pin: "", ok: true, isBio: false)
                    } else if pinMode & IW_PINMODE_BIO != 0, let biokey = self.checkIfBiokeyExists(pinMode: pinMode, alias: alias) {
                        self.sendOTP(alias: alias, pin: biokey, ok: true, isBio: true)
                    } else if pinMode & IW_PINMODE_CURRENT != 0 {
                        self.askForPinForPushOTP(alias: alias)
                    }
                } else {
                    self.showErrorFor(code: result)
                }
            }
        }
    }
    
    func checkIfBiokeyExists(pinMode: Int32, alias: String?) -> String? {
        guard let biokey = BiokeyService.sharedInstance.fetchBiokey() else {
            if let alias = alias, pinMode & IW_PINMODE_CURRENT != 0 {
                self.showAlert(title: "Biokey", message: "No Biokey registered or was not retrieved. Use Pin ?",
                               button1: (name: "Ok", action: {
                    _ in self.askForPinForPushOTP(alias: alias)
                }),
                               button2: (name: "Cancel", action: {
                    _ in self.navigationController?.popViewController(animated: true)
                }))
            } else {
                self.showAlert(title: "Biokey", message: "No Biokey registered or was not retrieved.", button1: (name: "Ok", action: {
                    _ in self.navigationController?.popViewController(animated: true)
                }), button2: nil)
            }
            return nil
        }
        return biokey
    }
    
    func sendOTP(alias: String, pin: String, ok: Bool, isBio: Bool) {
        let result = InweboService.sharedInstance.finalizePushOTP(alias: alias, pin: pin, ok: ok, isBio: isBio)
        DispatchQueue.main.async {
            if result == IW_ERR_OK {
                self.showAlert(title: "Push", message: "Push OTP sent")
            } else {
                self.showErrorFor(code: result)
            }
        }
    }


}
